<?php

namespace App\Http\Controllers;

use App\Models\CurrencyStocks;
use App\Models\Transaction;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function stats(Request $request)
    {
        $fromDate = $request->query('from_date');
        $toDate = $request->query('to_date', $fromDate ?? now()->toDateString());
        $currencyId = $request->query('currencyId');

        if (!$fromDate) {
            return response()->json(['message' => 'From date is required'], 400);
        }

        // Transactions in the date range
        $transactionsQuery = Transaction::query()
            ->whereBetween('transaction_date', [$fromDate, $toDate . ' 23:59:59'])
            ->when($currencyId, fn($q) => $q->where('currency_id', $currencyId))
            ->with('currency', 'stock');

        $transactions = $transactionsQuery->get();

        // Total Buy (RWF)
        $totalBuyRwf = $transactions->where('type', 'buy')->sum('total_rwf');

        // Total Sell (RWF)
        $totalSellRwf = $transactions->where('type', 'sell')->sum('total_rwf');

        // Profit: (sell_rate - buy_rate) * sell_amount
        $profit = 0;
        foreach ($transactions->where('type', 'sell') as $sell) {
            if ($sell->stock) {
                $buyRate = $sell->stock->buy_price;
                $sellRate = $sell->rate;
                $profit += ($sellRate - $buyRate) * $sell->amount;
            }
        }

        return response()->json([
            'totalBuyRwf' => $totalBuyRwf,
            'totalSellRwf' => $totalSellRwf,
            'profit' => $profit,
        ]);
    }

    public function currencyStocks(Request $request)
    {
        $stocks = CurrencyStocks::query()
            ->with('currency')
            ->get()
            ->groupBy('currency_id')
            ->map(function ($group) {
                return [
                    'currencyCode' => $group->first()->currency->code,
                    'totalAmount' => $group->sum('amount'),
                ];
            })->values();

        return response()->json($stocks);
    }

    public function prices(Request $request)
    {
        $currencyId = $request->query('currencyId');
        $date = $request->query('date', now()->toDateString());

        $latestBuy = Transaction::where('type', 'buy')
            ->where('transaction_date', '<=', $date . ' 23:59:59')
            ->when($currencyId, fn($q) => $q->where('currency_id', $currencyId))
            ->orderBy('transaction_date', 'desc')
            ->first();

        $latestSell = Transaction::where('type', 'sell')
            ->where('transaction_date', '<=', $date . ' 23:59:59')
            ->when($currencyId, fn($q) => $q->where('currency_id', $currencyId))
            ->orderBy('transaction_date', 'desc')
            ->first();

        return response()->json([
            'buyingPrice' => $latestBuy ? $latestBuy->rate : 0,
            'sellingPrice' => $latestSell ? $latestSell->rate : 0,
        ]);
    }
}
