<?php

namespace App\Http\Controllers;

use App\Models\CurrencyStocks;
use App\Models\Transaction;
use Illuminate\Http\Request;

class ReportController extends Controller
{
    public function daily(Request $request)
    {
        $startDate = $request->query('start_date');
        $endDate = $request->query('end_date', $startDate ?? now()->toDateString());
        $currencyId = $request->query('currency_id');
        $type = $request->query('type');

        // Validate dates
        if (!$startDate) {
            return response()->json(['message' => 'Start date is required'], 400);
        }

        // Starting Stock (before the range)
        $startingStockQuery = CurrencyStocks::query()
            ->where('loaded_at', '<', $startDate)
            ->when($currencyId, fn($q) => $q->where('currency_id', $currencyId))
            ->with('currency');

        $startingStock = $startingStockQuery->get()->map(function ($stock) {
            return [
                'currency_id' => $stock->currency_id,
                'currency_code' => $stock->currency->code,
                'amount' => $stock->amount,
                'buy_price' => $stock->buy_price
            ];
        });

        // Transactions within the range
        $transactionsQuery = Transaction::query()
            ->whereBetween('transaction_date', [$startDate, $endDate . ' 23:59:59'])
            ->when($currencyId, fn($q) => $q->where('currency_id', $currencyId))
            ->when($type, fn($q) => $q->where('type', $type))
            ->with('currency', 'stock');

        $transactions = $transactionsQuery->get();

        // Buys
        $buys = $transactions->where('type', 'buy')->map(function ($t) {
            return [
                'currency_id' => $t->currency_id,
                'currency_code' => $t->currency->code,
                'amount' => $t->amount,
                'buy_price' => $t->rate,
                'total_rwf' => $t->total_rwf
            ];
        })->values();

        // Sells with buy_rate and profit
        $sells = $transactions->where('type', 'sell')->map(function ($t) {
            $buyRate = $t->stock ? $t->stock->buy_price : 0;
            $profit = $t->stock ? ($t->rate - $buyRate) * $t->amount : 0;
            return [
                'currency_id' => $t->currency_id,
                'currency_code' => $t->currency->code,
                'amount' => $t->amount,
                'sell_price' => $t->rate,
                'buy_rate' => $buyRate,
                'total_rwf' => $t->total_rwf,
                'profit' => $profit
            ];
        })->values();

        // Ending Stock (after the range)
        $endingStockQuery = CurrencyStocks::query()
            ->where('loaded_at', '<=', $endDate . ' 23:59:59')
            ->when($currencyId, fn($q) => $q->where('currency_id', $currencyId))
            ->with('currency');

        $endingStock = $endingStockQuery->get()->map(function ($stock) use ($startDate, $endDate) {
            $soldAmount = Transaction::where('stock_id', $stock->id)
                ->where('type', 'sell')
                ->whereBetween('transaction_date', [$startDate, $endDate . ' 23:59:59'])
                ->sum('amount');
            return [
                'currency_id' => $stock->currency_id,
                'currency_code' => $stock->currency->code,
                'amount' => $stock->amount - $soldAmount,
                'buy_price' => $stock->buy_price
            ];
        })->filter(fn($s) => $s['amount'] > 0);

        // Totals
        $totalBuyRwf = $buys->sum('total_rwf');
        $totalSellRwf = $sells->sum('total_rwf');
        $profit = $sells->sum('profit');

        return response()->json([
            'start_date' => $startDate,
            'end_date' => $endDate,
            'starting_stock' => $startingStock,
            'buys' => $buys,
            'sells' => $sells,
            'ending_stock' => $endingStock,
            'total_buy_rwf' => $totalBuyRwf,
            'total_sell_rwf' => $totalSellRwf,
            'profit' => $profit
        ]);
    }
}
