<?php

namespace App\Http\Controllers;

use App\Models\CurrencyStocks;
use App\Models\Transaction;
use Carbon\Carbon;
use Illuminate\Http\Request;

class StockController extends Controller
{
    public function dailySummary(Request $request)
    {
        $date = Carbon::parse($request->query('date', now()))->startOfDay();
        $endOfDay = $date->copy()->endOfDay();

        // Starting stock: Stock before the selected date
        $startingStock = CurrencyStocks::where('loaded_at', '<', $date)
            ->groupBy('currency_id', 'buy_price')
            ->selectRaw('currency_id as currencyId, SUM(amount) as amount, buy_price as buyPrice')
            ->get();

        // Buys on the selected date
        $buys = Transaction::where('type', 'buy')
            ->whereBetween('transaction_date', [$date, $endOfDay])
            ->groupBy('currency_id', 'rate')
            ->selectRaw('currency_id as currencyId, SUM(amount) as amount, rate as buyPrice')
            ->get();

        // Sells on the selected date
        $sells = Transaction::where('type', 'sell')
            ->whereBetween('transaction_date', [$date, $endOfDay])
            ->groupBy('currency_id', 'rate')
            ->selectRaw('currency_id as currencyId, SUM(amount) as amount, rate as sellPrice')
            ->get();

        // Ending stock: Current stock as of now (or end of day)
        $endingStock = CurrencyStocks::groupBy('currency_id', 'buy_price')
            ->selectRaw('currency_id as currencyId, SUM(amount) as amount, buy_price as buyPrice')
            ->get();

        return response()->json([
            'startingStock' => $startingStock,
            'buys' => $buys,
            'sells' => $sells,
            'endingStock' => $endingStock
        ]);
    }

    public function index() {
        $stocks = CurrencyStocks::with('currency')
            ->where('amount', '>', 0) // Only return stocks with available amount
            ->get()
            ->map(function ($stock) {
                return [
                    'id' => $stock->id,
                    'currency_id' => $stock->currency_id,
                    'amount' => $stock->amount,
                    'buy_price' => $stock->buy_price,
                    'loaded_at' => $stock->loaded_at->toDateTimeString(),
                    'currency_code' => $stock->currency->code
                ];
            });

        return response()->json($stocks);
    }
}