<?php

namespace App\Http\Controllers;

use App\Models\Clients;
use App\Models\CurrencyStocks;
use App\Models\ExchangeRate;
use App\Models\Transaction;
use Illuminate\Http\Request;

class TransactionController extends Controller
{
    public function index()
    {
        $transactions = Transaction::with(['user', 'client', 'currency'])->get();
        return response()->json($transactions->map(function ($t) {
            return [
                'id' => $t->id,
                'userId' => $t->user_id,
                'userName' => $t->user->name,
                'clientId' => $t->client_id,
                'clientName' => $t->client->name,
                'phoneNumber' => $t->client->phone_number,
                'nationalId' => $t->client->national_id,
                'currencyId' => $t->currency_id,
                'currencyCode' => $t->currency->code,
                'stockId' => $t->stock_id,
                'stockAmount' => $t->stock ? $t->stock->amount : null,
                'stockLoadedAt' => $t->stock ? $t->stock->loaded_at : null,
                'type' => $t->type,
                'amount' => $t->amount,
                'rate' => $t->rate,
                'totalRwf' => $t->total_rwf,
                'transactionDate' => $t->transaction_date
            ];
        }));
    }

    public function show($id)
    {
        $transaction = Transaction::with(['user', 'client', 'currency'])->findOrFail($id);
        return response()->json([
            'id' => $transaction->id,
            'userId' => $transaction->user_id,
            'userName' => $transaction->user->name,
            'clientId' => $transaction->client_id,
            'clientName' => $transaction->client->name,
            'phoneNumber' => $transaction->client->phone_number,
            'nationalId' => $transaction->client->national_id,
            'currencyId' => $transaction->currency_id,
            'currencyCode' => $transaction->currency->code,
            'stockId' => $transaction->stock_id,
            'stockAmount' => $transaction->stock ? $transaction->stock->amount : null,
            'stockLoadedAt' => $transaction->stock ? $transaction->stock->loaded_at : null,
            'type' => $transaction->type,
            'amount' => $transaction->amount,
            'rate' => $transaction->rate,
            'totalRwf' => $transaction->total_rwf,
            'transactionDate' => $transaction->transaction_date
        ]);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'userId' => 'required|exists:users,id',
            'clientName' => 'required|string|max:255',
            'phoneNumber' => 'required|string|regex:/^[0-9]{10}$/',
            'nationalId' => 'required|string|max:16',
            'currencyId' => 'required|exists:currencies,id',
            'stockId' => 'nullable|exists:currency_stocks,id',
            'type' => 'required|in:buy,sell',
            'amount' => 'required|numeric|min:0',
            'rate' => 'required|numeric|min:0',
            'totalRwf' => 'required|numeric|min:0',
            'transactionDate' => 'required|date'
        ]);

        // First or create client
        $client = Clients::firstOrCreate(
            ['phone_number' => $data['phoneNumber']],
            ['name' => $data['clientName'], 'national_id' => $data['nationalId']]
        );

        if ($data['type'] === 'sell' && $data['stockId']) {
            $stock = CurrencyStocks::find($data['stockId']);
            if ($stock && $stock->amount >= $data['amount']) {
                $stock->amount -= $data['amount'];
                $stock->save();
            } else {
                return response()->json(['message' => 'Insufficient stock amount'], 400);
            }
        }

        $transaction = Transaction::create([
            'user_id' => $data['userId'],
            'client_id' => $client->id,
            'currency_id' => $data['currencyId'],
            'stock_id' => $data['stockId'],
            'type' => $data['type'],
            'amount' => $data['amount'],
            'rate' => $data['rate'],
            'total_rwf' => $data['totalRwf'],
            'transaction_date' => $data['transactionDate']
        ]);

        if ($data['type'] === 'buy') {
            $stock = CurrencyStocks::create([
                'currency_id' => $data['currencyId'],
                'amount' => $data['amount'],
                'buy_price' => $data['rate'],
                'loaded_at' => now()
            ]);
            $transaction->stock_id = $stock->id;
            $transaction->save();
        }

        return response()->json(['data' => $transaction], 201);
    }

    public function update(Request $request, $id)
    {
        $transaction = Transaction::findOrFail($id);
        $data = $request->validate([
            'userId' => 'required|exists:users,id',
            'clientName' => 'required|string|max:255',
            'phoneNumber' => 'required|string|regex:/^[0-9]{10}$/',
            'nationalId' => 'required|string|max:16',
            'currencyId' => 'required|exists:currencies,id',
            'stockId' => 'nullable|exists:currency_stocks,id',
            'type' => 'required|in:buy,sell',
            'amount' => 'required|numeric|min:0',
            'rate' => 'required|numeric|min:0',
            'totalRwf' => 'required|numeric|min:0',
            'transactionDate' => 'required|date'
        ]);

        // First or create client
        $client = Clients::firstOrCreate(
            ['phone_number' => $data['phoneNumber']],
            ['name' => $data['clientName'], 'national_id' => $data['nationalId']]
        );

        // Rollback old stock changes
        if ($transaction->type === 'sell' && $transaction->stock_id) {
            $oldStock = CurrencyStocks::find($transaction->stock_id);
            if ($oldStock) {
                $oldStock->amount += $transaction->amount;
                $oldStock->save();
            }
        } else if ($transaction->type === 'buy' && $transaction->stock_id) {
            CurrencyStocks::destroy($transaction->stock_id);
        }

        // Apply new stock changes
        if ($data['type'] === 'sell' && $data['stockId']) {
            $stock = CurrencyStocks::find($data['stockId']);
            if ($stock && $stock->amount >= $data['amount']) {
                $stock->amount -= $data['amount'];
                $stock->save();
            } else {
                return response()->json(['message' => 'Insufficient stock amount'], 400);
            }
        }

        $transaction->update([
            'user_id' => $data['userId'],
            'client_id' => $client->id,
            'currency_id' => $data['currencyId'],
            'stock_id' => $data['stockId'],
            'type' => $data['type'],
            'amount' => $data['amount'],
            'rate' => $data['rate'],
            'total_rwf' => $data['totalRwf'],
            'transaction_date' => $data['transactionDate']
        ]);

        if ($data['type'] === 'buy') {
            $stock = CurrencyStocks::create([
                'currency_id' => $data['currencyId'],
                'amount' => $data['amount'],
                'buy_price' => $data['rate'],
                'loaded_at' => now()
            ]);
            $transaction->stock_id = $stock->id;
            $transaction->save();
        }

        return response()->json(['data' => $transaction]);
    }

    public function destroy($id)
    {
        $transaction = Transaction::findOrFail($id);
        if ($transaction->type === 'sell' && $transaction->stock_id) {
            $stock = CurrencyStocks::find($transaction->stock_id);
            if ($stock) {
                $stock->amount += $transaction->amount;
                $stock->save();
            }
        } else if ($transaction->type === 'buy' && $transaction->stock_id) {
            CurrencyStocks::destroy($transaction->stock_id);
        }
        $transaction->delete();
        return response()->json(['message' => 'Transaction deleted']);
    }

    public function getLatestRate(Request $request)
    {
        $currencyId = $request->query('currencyId');
        if (!$currencyId) {
            return response()->json(['message' => 'Currency ID is required'], 400);
        }

        $latestRate = ExchangeRate::where('currency_id', $currencyId)
            ->orderBy('date', 'desc')
            ->first();

        if (!$latestRate) {
            return response()->json(['message' => 'No rate found for this currency'], 404);
        }

        return response()->json([
            'currencyId' => $latestRate->currency_id,
            'buyPrice' => $latestRate->buy_price,
            'sellPrice' => $latestRate->sell_price,
            'date' => $latestRate->date->toDateTimeString(),
        ]);
    }
}
