<?php

namespace App\Services;

use App\Models\Client;
use App\Models\CurrencyStocks;
use App\Models\ExchangeRate;
use App\Models\Transaction;

class TransactionService
{
    public function index() {
        return Transaction::with(['user', 'currency', 'stock'])
            ->orderBy('transaction_date', 'desc')
            ->get()
            ->map(function ($transaction) {
                return [
                    'id' => $transaction->id,
                    'user_id' => $transaction->user_id,
                    'user_name' => $transaction->user->name,
                    'currency_id' => $transaction->currency_id,
                    'currency_code' => $transaction->currency->code,
                    'stock_id' => $transaction->stock_id,
                    'phone_number' => $transaction->phone_number,
                    'national_id' => $transaction->national_id,
                    'type' => $transaction->type,
                    'amount' => $transaction->amount,
                    'rate' => $transaction->rate,
                    'total_rwf' => $transaction->total_rwf,
                    'transaction_date' => $transaction->transaction_date
                ];
            });
    }

    public function store(array $data) {
        $latestRate = ExchangeRate::where('currency_id', $data['currency_id'])
            ->latest('date')
            ->firstOrFail();

        $data['rate'] = $data['type'] === 'buy' ? $latestRate->buy_price : $latestRate->sell_price;

        if ($data['type'] === 'buy') {
            $stock = CurrencyStocks::create([
                'currency_id' => $data['currency_id'],
                'amount' => $data['amount'],
                'buy_price' => $data['rate'],
                'loaded_at' => now()
            ]);
        } else {
            // For sell, find the oldest stock (FIFO) and deduct
            $stock = CurrencyStocks::where('currency_id', $data['currency_id'])
                ->where('amount', '>', 0)
                ->orderBy('loaded_at', 'asc')
                ->firstOrFail();
            $stock->amount -= $data['amount'];
            $stock->save();
            $data['stock_id'] = $stock->id;
        }

        $transaction = Transaction::create($data);
        $transaction->load(['user', 'currency', 'stock']);
        return [
            'message' => 'Transaction recorded',
            'data' => [
                'id' => $transaction->id,
                'user_id' => $transaction->user_id,
                'user_name' => $transaction->user->name,
                'currency_id' => $transaction->currency_id,
                'currency_code' => $transaction->currency->code,
                'stock_id' => $transaction->stock_id,
                'phone_number' => $transaction->phone_number,
                'national_id' => $transaction->national_id,
                'type' => $transaction->type,
                'amount' => $transaction->amount,
                'rate' => $transaction->rate,
                'total_rwf' => $transaction->total_rwf,
                'transaction_date' => $transaction->transaction_date
            ]
        ];
    }

    public function update($id, array $data) {
        $transaction = Transaction::findOrFail($id);
        $transaction->update($data);
        $transaction->load(['user', 'currency']);
        return [
            'message' => 'Transaction updated',
            'data' => [
                'id' => $transaction->id,
                'user_id' => $transaction->user_id,
                'user_name' => $transaction->user->name,
                'currency_id' => $transaction->currency_id,
                'currency_code' => $transaction->currency->code,
                'phone_number' => $transaction->phone_number,
                'national_id' => $transaction->national_id,
                'type' => $transaction->type,
                'amount' => $transaction->amount,
                'rate' => $transaction->rate,
                'total_rwf' => $transaction->total_rwf,
                'transaction_date' => $transaction->transaction_date
            ]
        ];
    }

    public function destroy($id) {
        $transaction = Transaction::findOrFail($id);
        $transaction->delete();
        return [
            'message' => 'Transaction deleted successfully',
            'id' => $id
        ];
    }
}
